/*
	QTXCALC simple RPN Calculator
	Copyright (C) 1992-2012  Bernt Ribbum

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

//------------------------------------------------------------------------
// File for project XCALC
//-----------------------------------------------------------------------
// Various functions
//-----------------------------------------------------------------------

#include "util.h"
#include "xcalc.h"
#include "aschar.h"
#include <time.h>
#include <cstdlib>
#include <cstdio>
#include <complex>

// Convenience declarations
// Note: BC5.0 does not make proper LD values from decimal digits -
// therefore, we use bit patterns from BC3.1... (similar to M_PI_L)

#ifdef __HAVELD__
//const char radsindeg[] = {0xae,0xc8,0xe9,0x94,0x12,0x35,0xfa,0x8e,0xf9,0x3f};
//const char degsinrad[] = {0xc3,0xbd,0x0f,0x1e,0xd3,0xe0,0x2e,0xe5,0x04,0x40};
//const char m_pi[] = {0x35,0xc2,0x68,0x21,0xa2,0xda,0x0f,0xc9,0x00,0x40};
//const LD * const M_PI_L_PTR = (LD*)m_pi;
#endif

// A rare global variable!
QString g_helpdir;

//-----------------------------------------------------------------------
// Various functions for xcalc
//-----------------------------------------------------------------------
LD getreal(LC v) {
	return v.real();
}

LD getimag(LC v) {
	return v.imag();
}

bool isreal(LC v) {
	return v.imag()==0;
}

bool isregint(const LD &v)
{
	return isregintrange(v) && v == (LD)(qint64)v;
}

bool isregintrange(const LD &v)
{
	return v>=Register::minint() && v<=Register::maxint();
}

bool isint(LD v)
{
	return v==(qint64)v;
}

bool candms(const LD &v)
{
	LD a = floorl(fabsl(v));
	if (a>(LD)DMSMAXINT) return FALSE;
	return TRUE;
}

LD _hypot(const LD &x,const LD &y)
{
	return sqrt(x*x+y*y);
}

LD _hypotsq(const LD &x,const LD &y)
{
	return x*x+y*y;
}

long toLongRobust(QString s, bool &ok) // ok if any characters are used, or empty input (res=0)
{
	char *u = newasutf8(s);
	char *stop;
	long l = std::strtol(u,&stop,10);
	ok = s.isEmpty() || stop>u;
	delete[] u;
	return l;
}

LD toLDRobust(QString s, bool &ok) // ok if any characters are used, or empty input (res=0)
{
	if (s==".") { ok=true; return 0; }
	char *u = newasutf8(s);
	char *stop;
	LD ld = std::strtold(u,&stop);
	ok = s.isEmpty() || stop>u;
	delete[] u;
	return ld;
}

LC xcacos(LC x)
{
	return LC(M_PIl/2,0)-xcasin(x);
}

LC xcasin(LC x)
{
	LC i = LC(0,1);
	LC o = LC(1,0);
	return -i*log(i*x+sqrt(o-x*x));
}

LC xcatan(LC x)
{
	LC i = LC(0,1);
	LC o = LC(1,0);
	LC t = LC(2,0);
	return i*(log(o-i*x)-log(o+i*x))/t;
}

LC xcarcosh(LC x)
{
	LC o = LC(1,0);
	return log(x+sqrt(x+o)*sqrt(x-o));
}

LC xcarsinh(LC x)
{
	LC o = LC(1,0);
	return log(x+sqrt(x*x+o));
}

LC xcartanh(LC x)
{
	LC o = LC(1,0);
	LC t = LC(2,0);
	return log((o+x)/(o-x))/t;
}

//-----------------------------------------------------------------------
// Update 25/3/97: accept large negative power.
//-----------------------------------------------------------------------
LC intpow(LC y,qint64 x)
{
	if (y==LC(0) && x==0) {
		return NAN;
	}
	LC prod = y==LC(0)?LC(0,0):LC(1,0);	// initialise result to one unless y==0
	LC temp = y;				// temp for squaring where possible
	bool neg = (x<0);				// negative integer power
	x = abs(x);

	while (x) {
		// if odd power, multiply product with temp and decrement power by one
		if (x&1) {
			prod*=temp; // no range checking yet
			x--;
		}
		if (!x) break;		// done
		// now power is even - square temp and half power
		temp*=temp; // still no range checking
		x>>=1;
	}
	// result in prod. invert if negative power
	if (neg) {
		LC o = LC(1,0);
		return o/prod;
	}
	else {
		return prod;
	}
}

LC introot(LC y,qint64 x)
{
	//-----------------------------------------------------------------------
	//
	//-----------------------------------------------------------------------
	Register tmp(0,0,false);
	LD radius,angle;

	if (x==0)
		return NAN; // zero'th root is singularity

	if (y==LC(0,0)) {
		if (x<=0) return NAN; // neg root of zero is sing
		return 0; // pos root of zero is zero
	}

	if (isreal(y)) {
		// y is real: find correct length of y's radius, and set angle
		// depending on if x is odd (same angle) or even (divide angle by x)
		// therefore, odd roots never deliver complex results with real input.
		radius=abs(getreal(y));
		radius=pow(radius,1.0L/x);
		angle=atan2(getimag(y),getreal(y));
		if (!(x&1)) {
			tmp.set(radius * cos(angle/x),radius * sin(angle/x),false);
		}
		else {
			tmp.set(radius * cos(angle),radius * sin(angle),false);
		}
	}
	else {
		// y is complex: use clib's routine
		tmp = pow(y,1.0L/x);
	}

	return tmp;
}

QString chext(QString name, QString ext)
{
	int dot = name.lastIndexOf(EXTSEP);
	int sla = name.lastIndexOf(PATHSEP);
	if (dot>=0 && dot>sla)
		name=name.left(dot);
	return name+EXTSEP+ext;
}

QString nodir(QString name)
{
	int sla = name.lastIndexOf(PATHSEP);
	if (sla<0) return name;
	else return name.mid(sla+1);
}

QString appName()
{
	return nodir(appPath());
}

QString justdir(QString name)
{
	int sla = name.lastIndexOf(PATHSEP);
	if (sla<0) return name;
	else return name.left(sla+1);
}

QString appPath()
{
// if linux, look up the /proc/<PID>/exe link.
#ifdef __linux__
	QString link;
	char spath[128];
	memset(spath,0,128);
	link.sprintf("/proc/%d/exe",getpid());
	if (readlink(link.toUtf8().constData(),spath,128)>0) return spath;
	return "";
#else
#error OS not supported
#endif
}

QString snumber(qint64 v, WordLength wl, RadixType r) // instead of QString::number, with unsigned ints of specified wordlength
{
	char b65[65],*p=b65;
	memset(b65,0,65);
	intType val = v;
	quint8 i8=val.i8;
	quint16 i16=val.i16;
	quint32 i32=val.i32;
	quint64 i64=val.i64;
	switch(wl) {
	case wl8BIT:
		if (r==rtHEX) {
			sprintf(p,"%hhx",i8);
		} else if (r==rtOCTAL) {
			sprintf(p,"%hho",i8);
		} else if (r==rtBINARY) {
			p=b65+64; // fill backwards
			if (!i8) *--p='0';
			else while (i8) {
				*--p=i8&1?'1':'0';
				i8>>=1;
			}
		}
		break;
	case wl16BIT:
		if (r==rtHEX) {
			sprintf(p,"%hx",i16);
		} else if (r==rtOCTAL) {
			sprintf(p,"%ho",i16);
		} else if (r==rtBINARY) {
			p=b65+64; // fill backwards
			if (!i16) *--p='0';
			else while (i16) {
				*--p=i16&1?'1':'0';
				i16>>=1;
			}
		}
		break;
	case wl32BIT:
		if (r==rtHEX) {
			sprintf(p,"%x",i32);
		} else if (r==rtOCTAL) {
			sprintf(p,"%o",i32);
		} else if (r==rtBINARY) {
			p=b65+64; // fill backwards
			if (!i32) *--p='0';
			else while (i32) {
				*--p=i32&1?'1':'0';
				i32>>=1;
			}
		}
		break;
	case wl64BIT:
		if (r==rtHEX) {
			sprintf(p,"%llx",i64);
		} else if (r==rtOCTAL) {
			sprintf(p,"%llo",i64);
		} else if (r==rtBINARY) {
			p=b65+64; // fill backwards
			if (!i64) *--p='0';
			else while (i64) {
				*--p=i64&1?'1':'0';
				i64>>=1;
			}
		}
	}

	return (const char*)p;
}

void SetHelpDirDflt() {
// set help directory as html relative to application
	g_helpdir = justdir(appPath());
	if (!g_helpdir.endsWith("/"))
		g_helpdir += "/";
	g_helpdir += "html/";
}

void SetHelpDir(QString dir) // set another dir
{
	g_helpdir = dir;
}

void ShowHelp(QString file)
{
	QString path = g_helpdir + file;
#if defined __linux
	QString HELP = "xdg-open";

	pid_t pid;
	char *args[2];

	char *cmd = new char[HELP.toLatin1().length()+1];
	strcpy(cmd,HELP.toLatin1().constData());
	args[0] = cmd;
	args[1] = new char[path.toLatin1().length()+1];
	strcpy(args[1],path.toLatin1().constData());

	args[2]=0;
	pid=fork();
	if(!pid)
		execvp(cmd,args); // really weird API - duplicate cmd and args[0], or am I mistaken?
#elif defined __WINDOWS__
// shell execute should do it
ShellExecute(NULL,"open",path.toLatin1().constData(),NULL,NULL,SW_SHOWNORMAL);
#else
error "not supported OS"
#endif
}

void SendMail(QString rcpt)
{
	QString command = "mailto:";
#if defined __linux
	QString HELP = "xdg-open";

	pid_t pid;
	char *args[2];

	char *cmd = new char[command.toLatin1().length()+1];
	strcpy(cmd,HELP.toLatin1().constData());
	args[0] = cmd;
	args[1] = new char[rcpt.toLatin1().length()+1];
	strcpy(args[1],rcpt.toLatin1().constData());

	args[2]=0;
	pid=fork();
	if(!pid)
		execvp(cmd,args); // really weird API - duplicate cmd and args[0], or am I mistaken?
#elif defined __WINDOWS__
// shell execute should do it
ShellExecute(NULL,command,rcpt.toLatin1().constData(),NULL,NULL,SW_SHOWNORMAL);
#else
error "not supported OS"
#endif
}
