/*
	QTXCALC simple RPN Calculator
	Copyright (C) 2011  Bernt Ribbum

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef REGISTER_H
#define REGISTER_H

//#include <float.h>
#include "typedef.h"
//#include "xcalc.h"
#include <complex>

extern WordLength xcalc_wordLength;

enum ContentType {
	// preferred order (first nonzero counts) is ctFRACTION, ctINTEGER, ctCPX
	ctCPX,
	ctINTEGER,
	ctFRACTION,
};

struct intType {
	qint8 i8;
	qint16 i16;
	qint32 i32;
	qint64 i64;
public:
	intType(qint64 v=0) { i64=v; i32=(qint32)v; i16=(qint16)v; i8=(qint8)v; }
	// Problem here: how do I assure all shift operations are unsigned (e.g. not sign extend when bit 7 of i8 is set)
	// while also making sure translation to decimal/complex is always signed?
	//
	operator qint64() {
		switch (xcalc_wordLength) {
		case wl8BIT: return i8;
		case wl16BIT: return i16;
		case wl32BIT: return i32;
		default: return i64;
		}
	};
	operator quint64() {
		switch (xcalc_wordLength) {
		case wl8BIT: return (quint8)i8;
		case wl16BIT: return (quint16)i16;
		case wl32BIT: return (quint32)i32;
		default: return (quint64)i64;
		}
	};
	operator LD() { return (LD)(qint64)*this; }
	operator LC() { return (LC)(qint64)*this; }
};

class Register {
public:
	Register();
	Register(LD v1, LD v2=0,bool DMS=false);
	Register(qint64 l);
	Register(qint64 n, qint64 d);
	Register(LC v,bool DMS=false);
	Register &set(LD v1, LD v2=0, bool DMS=false) {return *this = Register(v1,v2,DMS); }
	Register &set(qint64 l) {return *this = Register(l); }
	Register &set(qint64 n,qint64 d) {return *this = Register(n,d); }
	Register &set(LC v, bool DMS=false) {return *this = Register(v,DMS);}
	void clear();
	operator LD();
	operator LC();
	LD realval() { return m_Real; }
	LD imagval() { return m_Imag; }
	qint64 ival() { return m_i.i64; }
	qint64 numval() { return m_num; }
	qint64 denval() { return m_den; }
	//void DS(); // downsize - fit in current wordsize
	//qint64 DS(qint64 v); // downsize - fit in current wordsize
	void reduce();
	void epstest();
	void nantest();
	bool isint();
	bool isregint();
	bool isregintrange();
	bool isfrac();
	bool isproperfrac();
	bool isproperint();
	bool iseven();
	bool iscomplex();
	bool ispropercomplex();
	bool iszero();
	bool isnegative();
	bool ispositive();
	bool isdms() { return m_DMS && candms(); }
	bool candms();
	LC ascomplex();
	LD asfloat();
	LD asreal();
	LD asimag();
	qint64 asint();
	qint64 asnum();
	qint64 asden();
	ContentType ct() { return m_ct; }
	void setDMS(bool v) { m_DMS=v; }
	// Maths functions (slowly, but they'll all be there!)
	Register &operator+=(Register &o);
	const Register operator+(Register &o) { return Register(*this)+=o; }
	Register &operator-=(Register &o);
	const Register operator-(Register &o) { return Register(*this)-=o; }
	Register &operator*=(Register &o);
	const Register operator*(Register &o) { return Register(*this)*=o; }
	Register &operator/=(Register &o);
	const Register operator/(Register &o) { return Register(*this)/=o; }
	Register &operator%=(Register &o);
	const Register operator%(Register &o) { return Register(*this)%=o; }
	const Register operator-();
	const Register sqrt();
	const Register log();
	const Register exp();
	const Register log10();
	const Register ten();
	const Register sin();
	const Register cos();
	const Register tan();
	const Register asin();
	const Register acos();
	const Register atan();
	const Register sinh();
	const Register cosh();
	const Register tanh();
	const Register arsinh();
	const Register arcosh();
	const Register artanh();
	const Register pow(Register &o);
	const Register root(Register &o);
	const Register And(Register &o);
	const Register Or(Register &o);
	const Register Xor(Register &o);
	const Register Not();
	const Register abs();
	const Register rcp();
	const Register fact();
	const Register random();
	void shl();		// in-place shift according to word size
	void shr();		// in-place shift according to word size
	void ashl();	// in-place ashift according to word size
	void ashr();	// in-place ashift according to word size
	void rotl();	// in-place rot according to word size
	void rotr();	// in-place rot according to word size
	static WordLength wordLength() { return xcalc_wordLength; }
	static void setWordLength(WordLength wl) { xcalc_wordLength=(WordLength)wl; }
	static qint64 minint();
	static qint64 maxint();
private:
	ContentType m_ct;
	LD m_Real, m_Imag;
	intType m_i;
	qint64 m_num,m_den;
	bool m_DMS; // for display
};

#endif // REGISTER_H
