/*
	QTXCALC simple RPN Calculator
	Copyright (C) 1992-2012  Bernt Ribbum

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "register.h"
#include "xcalc.h"
#include "util.h"

//RadixType Register::m_radix = DECIMAL;
//bool Register::m_cleanFrac = true;
//bool Register::m_invTrigAsDMS = false;
WordLength xcalc_wordLength = wl16BIT;

Register::Register() {
	m_ct = ctCPX;
	m_Real = 0;
	m_Imag = 0;
	m_i = 0;
	m_num = 0;
	m_den = 1;
	m_DMS = false;
	reduce();
}

Register::Register(LD v1, LD v2 ,bool DMS) {
	m_ct = ctCPX;
	m_Real = v1;
	m_Imag = v2;
	m_i = 0;
	m_num = 0;
	m_den = 1;
	m_DMS = DMS;
	reduce();
}

Register::Register(LC v, bool DMS) {
	m_ct = ctCPX;
	m_Real = v.real();
	m_Imag = v.imag();
	m_i = 0;
	m_num = 0;
	m_den = 1;
	m_DMS = DMS;
	reduce();
}

Register::Register(qint64 l) {
	m_ct = ctINTEGER;
	m_Real = 0;
	m_Imag = 0;
	m_i = l;
	m_num = 0;
	m_den = 1;
	m_DMS = false;
	reduce();
}

Register::Register(qint64 n, qint64 d) {
	m_ct = ctFRACTION;
	m_Real = 0;
	m_Imag = 0;
	m_i = 0;
	m_num = n;
	m_den = d?d:1;
	m_DMS = false;
	reduce();
}

void Register::clear() {
	m_ct = ctINTEGER;
	m_Real = 0;
	m_Imag = 0;
	m_i = 0;
	m_num = 0;
	m_den = 1;
	m_DMS = false;
}

Register::operator LD() {
	return asfloat();
}

Register::operator LC() {
	return ascomplex();
}

LD Register::asfloat() {
	switch (m_ct) {
	case ctINTEGER: return (LD)m_i;
	case ctFRACTION: return (LD)m_num/(LD)m_den;
	//case ctCPX: return hypotl(m_Real,m_Imag);// std::abs(m_cpx);
	case ctCPX: return m_Real;
	default: return 42;
	}
}

LC Register::ascomplex() {
	switch (m_ct) {
	case ctINTEGER: return (LD)m_i;
	case ctFRACTION: return (LC)m_num/(LD)m_den;
	case ctCPX: return LC(m_Real,m_Imag);
	default: return 42;
	}
}

LD Register::asreal() {
	switch (m_ct) {
	case ctINTEGER: return m_i;
	case ctFRACTION: return (LD)m_num/(LD)m_den;
	case ctCPX: return m_Real;
	default: return 42;
	}
}

LD Register::asimag() {
	switch (m_ct) {
	case ctINTEGER: return 0;
	case ctFRACTION: return 0;
	case ctCPX: return m_Imag;
	default: return 42;
	}
}

qint64 Register::asint() {
	switch (m_ct) {
	case ctINTEGER: return m_i;
	case ctFRACTION: return m_num/m_den;
	case ctCPX: return (qint64)hypotl(m_Real,m_Imag);// std::abs(m_cpx);
	default: return 42;
	}
}

qint64 Register::asnum() {
	if (isproperfrac()) return m_num;
	else return asint();
}

qint64 Register::asden() {
	if (isproperfrac()) return m_den;
	else return 1;
}

void Register::epstest() {
	// check relative size of real/imag parts, zeroing out a too small one
	LD r = fabsl(m_Real);
	LD i  = fabsl(m_Imag);
	if (i<r && i/r<64*LDBL_EPSILON) m_Imag = 0;
	if (r<i && r/i<64*LDBL_EPSILON) m_Real = 0;
}

void Register::nantest() {
	if (!isfinite(m_Real) && isnan(m_Imag)) m_Imag=0; // inf,nan -> inf,0
}

/*
void Register::DS() {
	switch (xcalc_wordLength) {
	case WL8BIT: m_i=m_i.i8;
	case WL16BIT: m_i=m_i.i16;
	case WL32BIT: m_i=m_i.i32;
	case WL64BIT: m_i=m_i.i64;
	}
}

qint64 Register::DS(qint64 v) { // DS = downsize
	switch (xcalc_wordLength) {
	case WL8BIT: return (qint8)v;
	case WL16BIT: return (qint16)v;
	case WL32BIT: return (qint32)v;
	case WL64BIT: return (qint64)v; // should have no effect, but cast for completeness
	}
}
*/

qint64 Register::minint(){
	switch (xcalc_wordLength) {
	case wl8BIT: return SCHAR_MIN;
	case wl16BIT: return SHRT_MIN;
	case wl32BIT: return INT_MIN;
	case wl64BIT: return LONG_LONG_MIN;
	}
}

qint64 Register::maxint(){
	switch (xcalc_wordLength) {
	case wl8BIT: return SCHAR_MAX;
	case wl16BIT: return SHRT_MAX;
	case wl32BIT: return INT_MAX;
	case wl64BIT: return LONG_LONG_MAX;
	}
}

void Register::reduce() {
	// Find correct number representation:
	// - Promote from fraction or real to int if possible,
	// - Downsize integers to current wordsize
	// - Reduce if fraction,
	// - Choose first nonzero value using INTEGER, FRACTION, CPX in that order. (INTEGER if zero)
	// - Check real/imag relative sizes, zeroing a too small value
	// - Zero imag part if NaN (pow does that if real part->inf)
	// - Keep m_DMS if present.
	// - Let complex part always contain shadow copy of INT or FRAC value.
	// Called automatically from constructor.

	if ((qint64)m_i!=0) m_ct = ctINTEGER;
	else if (m_num!=0) m_ct = ctFRACTION;
	else m_ct = ctCPX;

	if (m_ct==ctCPX && imagval()==0 && ::isregint(realval())) {
		m_i=(qint64)realval();
		m_ct=ctINTEGER;
	}
	if (m_ct==ctFRACTION) {
		qint64 n, d, r;

		// We could end up here with a number such as "1/-2", so we check
		if (m_den < 0) {
			m_den = -m_den;
			m_num = -m_num;
		}

		n = m_num;
		d = m_den;

		// Only come here with proper (nonzero) fractions.
		if (n == 0) { // then m_i is 0 as well. Number is INT
			m_Real=0;
			m_Imag=0;
			m_den = 1;
			m_ct = ctINTEGER;
			return; // No fraction. Should never happen
		}
		if (n<0) n = -n;

		while (d>0) {
			r = n%d;
			n = d;
			d = r;
		}

		m_num /= n;
		m_den /= n;

		if (m_den == 1) {
			m_i = m_num;
			m_Real = 0;
			m_Imag = 0;
			m_num = 0;
			m_ct = ctINTEGER;
		}
	}
	// Make shadow copy of ctINTEGER and ctFRACTION values (don't think it is needed)
	if (m_ct==ctINTEGER) m_Real = m_i;
	if (m_ct==ctFRACTION) m_Real = (LD)m_num/(LD)m_den;
	if (m_ct==ctCPX) { // eps check
		epstest(); // test relative real/imag size, zeroing out negligible value
		nantest(); // zero nan imag part
	}
} // reduce

bool Register::isproperint() // zero considered int here (same as isint)
{
	return m_ct==ctINTEGER;
}

bool Register::isregint()
{
	return isint() && isregintrange();
}

bool Register::isregintrange()
{
	return (qint64)this->m_i>=Register::minint() && (qint64)this->m_i<=Register::maxint();
}

bool Register::isint() // zero considered int here
{
	return m_ct==ctINTEGER;
}

bool Register::isproperfrac()
{
	return m_ct==ctFRACTION;
}

bool Register::isfrac() // zero and int considered fractions here
{
	return isproperfrac() || isint();
}

bool Register::iseven()
{
	return isint() && ((qint64)realval()&0x01)==0;
}

bool Register::iscomplex() // zero considered complex here
{
	return m_ct==ctCPX || iszero();
}

bool Register::ispropercomplex() // imag part !=0
{
	return m_ct==ctCPX && m_Imag!=0;
}

bool Register::iszero()
{
	return m_ct==ctINTEGER && (qint64)m_i==0;
}

bool Register::isnegative() // and nonzero
{
	if (m_ct==ctCPX) return imagval()==0.0 && realval()<0.0;
	else if (m_ct==ctFRACTION) return m_num<0;
	else return (qint64)m_i<0;
}

bool Register::ispositive() // and nonzero
{
	if (m_ct==ctCPX) return imagval()==0.0 && realval()>0.0;
	else if (m_ct==ctFRACTION) return m_num>0;
	else return (qint64)m_i>0;
}

bool Register::candms()
{
	LD a = (LD)*this;
	return a<=(LD)DMSMAXINT;
}

// Maths operations - here is the actual work!
// Note that there are no errors any more - invalid results are Inf or NaN.
// Binary results just saturate/wrap.

Register &Register::operator+=(Register &o) {
	if (((isproperfrac()&&o.isfrac())||(isfrac()&&o.isproperfrac()))
			&&::isint((LD)asnum()*o.asden()+(LD)o.asnum()*asden())
			&&::isint((LD)asden()*o.asden()))
		set(asnum()*o.asden()+o.asnum()*asden(),asden()*o.asden());
	else
		set(ascomplex()+o.ascomplex(),m_DMS||o.m_DMS);
		//set(asreal()+o.asreal(),asimag()+o.asimag(),m_DMS||o.m_DMS);
	return *this;
}

Register &Register::operator-=(Register &o) {
	if (((isproperfrac()&&o.isfrac())||(isfrac()&&o.isproperfrac()))
			&&::isint((LD)asnum()*o.asden()-(LD)o.asnum()*asden())
			&&::isint((LD)asden()*o.asden()))
		set(asnum()*o.asden()-o.asnum()*asden(),asden()*o.asden());
	else
		set(ascomplex()-o.ascomplex(),m_DMS||o.m_DMS);
		//set(asreal()-o.asreal(),asimag()-o.asimag(),m_DMS||o.m_DMS);
	return *this;
}

Register &Register::operator*=(Register &o) {
	if (((isproperfrac()&&o.isfrac())||(isfrac()&&o.isproperfrac()))
			&&::isint((LD)asnum()*o.asnum())
			&&::isint((LD)asden()*o.asden()))
		set(asnum()*o.asnum(),asden()*o.asden());
	else
		set(ascomplex()*o.ascomplex(),m_DMS);
	return *this;
}

Register &Register::operator/=(Register &o) {
	if (((isproperfrac()&&o.isfrac())||(isfrac()&&o.isproperfrac()))
			&&::isint((LD)asnum()*o.asden())
			&&::isint((LD)asden()*o.asnum()))
		set(asnum()*o.asden(),asden()*o.asnum());
	else {
		set(ascomplex()/o.ascomplex(),m_DMS);
	}
	return *this;
}

Register &Register::operator%=(Register &o) {
	// This makes little sense for complex numbers. (returns same as division)
	// TODO: Fractions could be done though.
	if (((isproperfrac()&&o.isfrac())||(isfrac()&&o.isproperfrac()))
			&&::isint((LD)asnum()*o.asden())
			&&::isint((LD)asden()*o.asnum()))
		set(asnum()*o.asden(),asden()*o.asnum());
	else {
		set(ascomplex()/o.ascomplex(),m_DMS);
	}
	return *this;
}

const Register Register::operator-() {
	Register r(*this);
	if (r.isproperfrac()&&(-r.asnum()==-(LD)r.asnum()))
		r.set(-asnum(),asden());
	else
		r.set(-ascomplex(),m_DMS);
	return r;
}

const Register Register::sqrt() {
	LC t(ascomplex());
	return std::sqrt(t);
}

const Register Register::log() {
	LC t(ascomplex());
	return std::log(t);
}

const Register Register::exp() {
	LC t(ascomplex());
	return std::exp(t);
}

const Register Register::log10() {
	LC t(ascomplex());
	return std::log10(t);
}

const Register Register::ten() {
	LC t(10,0);
	LC x(ascomplex());
	return std::pow(t,x);
}

const Register Register::pow(Register &o) {
	LC y(ascomplex());
	if (o.isint()) {
		qint64 x=o.asint();
		return intpow(y,x);
	} else {
		LC x(o.ascomplex());
		return std::pow(y,x);
	}
}

const Register Register::root(Register &o) {
	LC y(ascomplex());
	if (o.isint()) {
		qint64 x=o.asint();
		return introot(y,x);
	} else {
		LC one(1,0);
		LC x(o.ascomplex());
		return std::pow(y,one/x);
	}
}

const Register Register::And(Register &o) {
	return asint()&o.asint();
}

const Register Register::Or(Register &o) {
	return asint()|o.asint();
}

const Register Register::Xor(Register &o) {
	return asint()^o.asint();
}

const Register Register::Not() {
	return ~asint();
}

const Register Register::sin() {
	return std::sin(ascomplex());
}

const Register Register::cos() {
	return std::cos(ascomplex());
}

const Register Register::tan() {
	return std::tan(ascomplex());
}

const Register Register::asin() {
	return Register(xcasin(ascomplex()),false/*Register::invTrigAsDMS()*/);
}

const Register Register::acos() {
	return Register(xcacos(ascomplex()),false/*Register::invTrigAsDMS()*/);
}

const Register Register::atan() {
	return Register(xcatan(ascomplex()),false/*Register::invTrigAsDMS()*/);
}

const Register Register::sinh() {
	return std::sinh(ascomplex());
}

const Register Register::cosh() {
	return std::cosh(ascomplex());
}

const Register Register::tanh() {
	return std::tanh(ascomplex());
}

const Register Register::arsinh() {
	return xcarsinh(ascomplex());
}

const Register Register::arcosh() {
	return xcarcosh(ascomplex());
}

const Register Register::artanh() {
	return xcartanh(ascomplex());
}

const Register Register::abs(){
	return std::abs(ascomplex());
}

const Register Register::rcp(){
	return LC(1,0)/ascomplex();
}

const Register Register::fact(){
	qint64 i=asint();
	LD m = 1.0;
	while (i) m*=--i;
	return Register(m,0,false);
}
const Register Register::random(){
	m_ct = ctCPX;
	m_Real = drand48();
	m_Imag = 0;
	return *this;
}

void Register::shl(){
	switch (xcalc_wordLength) {
	case wl8BIT: m_i.i8 <<= 1; break;
	case wl16BIT: m_i.i16 <<= 1; break;
	case wl32BIT: m_i.i32 <<= 1; break;
	case wl64BIT: m_i.i64 <<= 1; break;
	}
}

void Register::shr(){
	switch (xcalc_wordLength) {
	case wl8BIT: m_i.i8= (quint8)m_i.i8 >> 1; break;
	case wl16BIT: m_i.i16= (quint16)m_i.i16>> 1; break;
	case wl32BIT: m_i.i32= (quint32)m_i.i32>> 1; break;
	case wl64BIT: m_i.i64= (quint64)m_i.i64>> 1; break;
	}
}

void Register::ashl(){
	switch (xcalc_wordLength) {
	case wl8BIT: m_i.i8 <<= 1; break;
	case wl16BIT: m_i.i16 <<= 1; break;
	case wl32BIT: m_i.i32 <<= 1; break;
	case wl64BIT: m_i.i64 <<= 1; break;
	}
}

void Register::ashr(){
	switch (xcalc_wordLength) {
	case wl8BIT: m_i.i8 >>= 1; break;
	case wl16BIT: m_i.i16 >>= 1; break;
	case wl32BIT: m_i.i32 >>= 1; break;
	case wl64BIT: m_i.i64 >>= 1; break;
	}
}

void Register::rotl(){
	switch (xcalc_wordLength) {
	case wl8BIT: m_i.i8 = (quint8)m_i.i8<<1 | (quint8)m_i.i8>>7; break;
	case wl16BIT: m_i.i16 = (quint16)m_i.i16<<1 | (quint16)m_i.i16>>15; break;
	case wl32BIT: m_i.i32 = (quint32)m_i.i32<<1 | (quint32)m_i.i32>>31; break;
	case wl64BIT: m_i.i64 = (quint64)m_i.i64<<1 | (quint64)m_i.i64>>63; break;
	}
}

void Register::rotr(){
	switch (xcalc_wordLength) {
	case wl8BIT: m_i.i8 = (quint8)m_i.i8>>1 | (quint8)m_i.i8<<7; break;
	case wl16BIT: m_i.i16 = (quint16)m_i.i16>>1 | (quint16)m_i.i16<<15; break;
	case wl32BIT: m_i.i32 = (quint32)m_i.i32>>1 | (quint32)m_i.i32<<31; break;
	case wl64BIT: m_i.i64 = (quint64)m_i.i64>>1 | (quint64)m_i.i64<<63; break;
	}
}
